<?php

namespace App\Http\Controllers\Admin;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Order;
use App\Models\OrderItem;
use App\Models\OrderItemDetail;
use App\Models\OrderTransaction;
use App\Models\Item;
use App\Models\ItemDetail;
use App\Models\ItemStore;
use App\Models\Customer;
use App\Models\CustomerLinkPackage;
use App\Models\Package;
use App\Models\OrderUser;
use App\Models\OrderHistory;
use App\Models\PaymentMethod;
use App\Models\OrderCustomer;
use App\Models\OrderStatus;
use App\Models\CustomerPackage;
use App\Models\PackageDetail;
use App\Models\PackageItem;
use App\Models\Setting;
use App\Models\PermissionToUser;
use Illuminate\Support\Facades\Hash;
use Illuminate\Contracts\Database\Eloquent\Builder;
use Barryvdh\DomPDF\Facade\Pdf;
use PHPUnit\Event\Telemetry\System;

class OrderController extends Controller
{
    public $role_id;
    public $permission = false;
    public $permission_type;

    public function __construct()
    {
        $permissions = PermissionToUser::where('role_id', session('role_id'))->get();

        $permission_type = [];

        foreach($permissions as $permission) {
            $permission_type[] = array(
                'permission'    => $permission->permission,
            );
        }

        if (array_search('orders', array_column($permission_type, 'permission'))) {
            $this->permission = true;
        } else {
            $this->permission = false;
        }

        if (array_search('order_show', array_column($permission_type, 'permission'))) {
            $this->permission_type['order_show'] = true;
        } else {
            $this->permission_type['order_show'] = false;
        }

        if (array_search('order_create', array_column($permission_type, 'permission'))) {
            $this->permission_type['order_create'] = true;
        } else {
            $this->permission_type['order_create'] = false;
        }

        if (array_search('order_update', array_column($permission_type, 'permission'))) {
            $this->permission_type['order_update'] = true;
        } else {
            $this->permission_type['order_update'] = false;
        }

        if (array_search('order_delete', array_column($permission_type, 'permission'))) {
            $this->permission_type['order_delete'] = true;
        } else {
            $this->permission_type['order_delete'] = false;
        }

        if(!session('role_id')) {
            return redirect()->route('login');
        }
        $this->role_id = session('role_id');

    }

    public function index()
    {
        $orders =   Order::with(['order_customers', 'user'])
                    ->addSelect([
                        'amount_paid' => OrderTransaction::selectRaw('sum(payment_amount) as paid')
                        ->whereColumn('order_id', 'orders.id')
                        ->groupBy('order_id')
                    ])
                    ->orderBy('order_date', 'DESC')
                    ->get();
        
        $permission_type['order_show'] = $this->permission_type['order_show'];
        $permission_type['order_create'] = $this->permission_type['order_create'];
        $permission_type['order_update'] = $this->permission_type['order_update'];
        $permission_type['order_delete'] = $this->permission_type['order_delete'];

        return response()->json([
            'orders' => $orders,
            'permission'    => $this->permission,
            'permission_type'   => $permission_type,
        ]);
    }

    public function create()
    {
        $payment_methods = PaymentMethod::with(['detail'])->where('status', '1')->orderBy('sort', 'ASC')->get();
        $order_statuses = OrderStatus::with(['detail'])->where('status', 1)->orderBy('sort', 'ASC')->get();
        $items = Item::with(['detail'])->where('status', '1')->orderBy('sort', 'ASC')->get();
        $packages = Package::with(['detail'])->where('status', '1')->get();
        $customers = Customer::where('status', '1')->get();
        
        $permission_type['order_create'] = $this->permission_type['order_create'];

        return response()->json([
            'payment_methods' => $payment_methods,
            'items' => $items,
            'packages' => $packages,
            'customers' => $customers,
            'order_statuses' => $order_statuses,
            'permission_type'   => $permission_type,
        ]);
    }

    public function store(Request $request)
    {
        if($this->permission_type['order_create'] != true) {
            return response()->json(['success' => false]);
        }
        $data['total'] = $request->input('grand_total');
        $data['payment_option'] = $request->input('payment_option');
        $data['payment_method_id'] = $request->input('payment_method_id');

        $payment_method = PaymentMethod::with(['detail'])->where('id', $request->input('payment_method_id'))->first();
        $data['payment_method'] = $payment_method->detail->title;

        $data['user_id'] = session('user_id');
        $data['comment'] = $request->input('comment');
        $data['order_type'] = 'custom';
        $data['order_date'] = $request->input('order_date');

        $status_id = $request->input('order_status_id');

        $status = OrderStatus::with(['detail'])->where('id', $status_id)->first();

        $data['status'] = $status->detail->title ?? NULL;
        $data['customers'] = $request->input('order_customers');

        $order = Order::create($data);

        $customers = json_decode($request->input('order_customers'));

        foreach($customers as $customer) {
            $customer_info['order_id'] = $order->id;

            if($customer->type == 'existing_customer') {
                $customer_data = Customer::where('id', $customer->id)->first();
                $customer_info['customer_id'] = $customer_data->id;
                $customer_info['company'] = $customer_data->company;
                $customer_info['name'] = $customer_data->name;
                $customer_info['email'] = $customer_data->email;
                $customer_info['telephone'] = $customer_data->telephone;
            }

            if($customer->type == 'new_customer') {
                $customer_info['company'] = $customer->company;
                $customer_info['name'] = $customer->name;
                $customer_info['email'] = $customer->customer_email;
                $customer_info['country_code'] = $customer->country_code ?? '000';
                $customer_info['telephone'] = $customer->telephone;
                $customer_info['password'] = Hash::make($customer->telephone);
                $customer_info['language_id'] = '0';
                $customer_info['status'] = '1';

                $new_customer_data = Customer::create($customer_info);
                $customer_info['customer_id'] = $new_customer_data->id;

            }

            OrderCustomer::create($customer_info);

        }

        $order_transaction['order_id'] = $order->id;
        $order_transaction['payment_method_id'] = $request->input('payment_method_id');
        $order_transaction['payment_method'] = $payment_method->detail->title;
        $order_transaction['payment_amount'] = $request->input('payment_amount');

        OrderTransaction::create($order_transaction);

        $items = json_decode($request->input('order_items'));

        foreach($items as $item) {
            $order_item['order_id'] = $order->id;
            $order_item['service_id'] = $item->id ?? 0;
            $order_item['detail'] = $item->detail;
            $order_item['type'] = $item->type;
            $order_item['quantity'] = $item->quantity;
            $order_item['unit_price'] = $item->unit_price;
            $order_item['total'] = $item->total;

            $order_item_result = OrderItem::create($order_item);

            if($item->type == 'item') {
                $item_details = ItemDetail::where('item_id', $item->id)->get();

                foreach($item_details as $item_detail) {
                    $order_item_detail['order_id'] = $order->id;
                    $order_item_detail['order_item_id'] = $order_item_result->id;
                    $order_item_detail['title'] = $item_detail->title;
                    $order_item_detail['language_locale'] = $item_detail->language_locale;
                    OrderItemDetail::create($order_item_detail);
                }
                
            }

            if($item->type == 'package') {
                $package_details = PackageDetail::where('package_id', $item->id)->get();

                foreach($package_details as $package_detail) {
                    $order_item_detail['order_id'] = $order->id;
                    $order_item_detail['order_item_id'] = $order_item_result->id;
                    $order_item_detail['title'] = $package_detail->title;
                    $order_item_detail['language_locale'] = $package_detail->language_locale;
                    OrderItemDetail::create($order_item_detail);
                }

                $packageItems = PackageItem::where('package_id', $item->id)->get();

                foreach($packageItems as $packageItem) {

                    $CustomerPackage['branch_type'] = 'all';
                    
                    if($packageItem->type == 'item') 
                    {
                        $item = Item::with(['detail'])->where('id', $packageItem->item_id)->first();
                        
                        $CustomerPackage['title'] = $item->detail->title ?? "";
                        
                        if($item->store_type == 'custom') {
                            $CustomerPackage['branch_type'] = 'custom';
                        }
                    }

                    if($packageItem->type == 'text') 
                    {
                        $CustomerPackage['title'] = $packageItem->detail->title;
                    }

                    $CustomerPackage['order_id'] = $order->id;
                    $CustomerPackage['package_item_id'] = $packageItem->id;
                    $CustomerPackage['item_id'] = $item->id;
                    $CustomerPackage['price'] = $item->price;
                    $CustomerPackage['type'] = $item->type;
                    
                    $CustomerPackage['times'] = $packageItem->times;
                    $CustomerPackage['time_used'] = 0;
                    $CustomerPackage['time_remain'] = $packageItem->times;
                    $CustomerPackage['expiry_date'] = date("Y-m-d", strtotime("+".$packageItem->duration." Months"));
                    $customerPackageInfo = CustomerPackage::create($CustomerPackage);

                    $order_customers = OrderCustomer::where('order_id', $order->id)->get();

                    foreach($order_customers as $order_customer) {
                        $order_customer_info['order_id'] = $order_customer->order_id;
                        $order_customer_info['customer_package_id'] = $customerPackageInfo->id;
                        $order_customer_info['customer_id'] = $order_customer->customer_id;
                        CustomerLinkPackage::create($order_customer_info);
                    }

                }

            }
        }

        // $this->pdf_save($order->id);

        return response()->json([
            'message' => 'success',
        ]);

    }

    public function show(string $id)
    {
        $order = Order::with(['order_customers', 'order_items', 'transactions'])->where('id', $id)->first();
        $payment_methods = PaymentMethod::with(['detail'])->where('status', '1')->orderBy('sort', 'ASC')->get();
        $items = Item::with(['detail'])->where('status', '1')->orderBy('sort', 'ASC')->get();
        $customers = Customer::where('status', 1)->get();
        $order_statuses = OrderStatus::with(['detail'])->where('status', 1)->orderBy('sort', 'ASC')->get();
        $packages = Package::with(['detail'])->where('status', '1')->get();
        $permission_type['order_update'] = $this->permission_type['order_update'];

        return response()->json([
            'order'             => $order,
            'payment_methods'   => $payment_methods,
            'items'             => $items,
            'packages'       => $packages,
            'customers'         => $customers,
            'order_statuses'    => $order_statuses,
            'permission_type'   => $permission_type,
        ]);
    }

    public function edit(string $id)
    {
        $order = Order::with(['order_customers', 'order_items', 'transactions'])->where('id', $id)->first();
        $payment_methods = PaymentMethod::with(['detail'])->where('status', '1')->orderBy('sort', 'ASC')->get();
        $items = Item::with(['detail'])->where('status', '1')->orderBy('sort', 'ASC')->get();
        $customers = Customer::where('status', 1)->get();
        $order_statuses = OrderStatus::with(['detail'])->where('status', 1)->orderBy('sort', 'ASC')->get();
        $packages = Package::with(['detail'])->where('status', '1')->get();
        $permission_type['order_update'] = $this->permission_type['order_update'];

        return response()->json([
            'order'             => $order,
            'payment_methods'   => $payment_methods,
            'items'             => $items,
            'packages'       => $packages,
            'customers'         => $customers,
            'order_statuses'    => $order_statuses,
            'permission_type'   => $permission_type,
        ]);

    }

    public function update(Request $request, string $id)
    {
        if($this->permission_type['order_update'] != true) {
            return response()->json(['success' => false]);
        }
        $order = Order::find($id);
        $data['comment'] = $request->input('comment');
        $data['total'] = $request->input('sub_total');
        $data['status'] = $request->input('status');
        $order->update($data);

        OrderCustomer::where('order_id', $id)->delete();

        $customers = json_decode($request->input('order_customers'));

        foreach($customers as $customer) {
            $customer_info['order_id'] = $id;

            if($customer->type == 'existing_customer') {
                $customer_data = Customer::where('id', $customer->customer_id)->first();
                if($customer_data) {
                    $customer_info['customer_id'] = $customer_data->id;
                    $customer_info['company'] = $customer_data->company;
                    $customer_info['name'] = $customer_data->name;
                    $customer_info['email'] = $customer_data->email;
                    $customer_info['telephone'] = $customer_data->telephone;
                }
                else {
                    $customer_info['customer_id'] = 0;
                    $customer_info['company'] = 'no data';
                    $customer_info['name'] = 'no data';
                    $customer_info['email'] = 'no data';
                    $customer_info['telephone'] = 'no data';
                }
            }

            if($customer->type == 'new_customer') {
                $customer_info['company'] = $customer->company;
                $customer_info['name'] = $customer->name;
                $customer_info['email'] = $customer->customer_email;
                $customer_info['country_code'] = '';
                $customer_info['telephone'] = $customer->telephone;
                $customer_info['password'] = Hash::make($customer->telephone);
                $customer_info['language_id'] = '86';
                $customer_info['status'] = '1';

                $new_customer_data = Customer::create($customer_info);
                $customer_info['customer_id'] = $new_customer_data->id;
            }

            OrderCustomer::create($customer_info);
        }

        OrderItem::where('order_id', $id)->delete();
        OrderItemDetail::where('order_id', $id)->delete();
        CustomerPackage::where('order_id', $id)->delete();
        CustomerLinkPackage::where('order_id', $id)->delete();

        $items = json_decode($request->input('order_items'));

        foreach($items as $item) {
            $order_item['order_id'] = $id;
            $order_item['service_id'] = $item->id ?? 0;
            $order_item['detail'] = $item->detail;
            $order_item['type'] = $item->type;
            $order_item['quantity'] = $item->quantity;
            $order_item['unit_price'] = $item->unit_price;
            $order_item['total'] = $item->total;

            $order_item_result = OrderItem::create($order_item);

            if($item->type == 'item') {
                $item_details = ItemDetail::where('item_id', $item->id)->get();

                foreach($item_details as $item_detail) {
                    $order_item_detail['order_id'] = $id;
                    $order_item_detail['order_item_id'] = $order_item_result->id;
                    $order_item_detail['title'] = $item_detail->title;
                    $order_item_detail['language_locale'] = $item_detail->language_locale;
                    OrderItemDetail::create($order_item_detail);
                }
                
            }

            if($item->type == 'package') {
                $package_details = PackageDetail::where('package_id', $item->id)->get();

                foreach($package_details as $package_detail) {
                    $order_item_detail['order_id'] = $id;
                    $order_item_detail['order_item_id'] = $order_item_result->id;
                    $order_item_detail['title'] = $package_detail->title;
                    $order_item_detail['language_locale'] = $package_detail->language_locale;
                    OrderItemDetail::create($order_item_detail);
                }

                $packageItems = PackageItem::where('package_id', $item->id)->get();

                foreach($packageItems as $packageItem) {

                    $CustomerPackage['branch_type'] = 'all';

                    if($packageItem->type == 'item') 
                    {
                        $item = Item::with(['detail'])->where('id', $packageItem->item_id)->first();
                        $CustomerPackage['title'] = $item->detail->title;
                        
                        if($item->store_type == 'custom') {
                            $CustomerPackage['branch_type'] = 'custom';
                        }
                        
                    }

                    if($packageItem->type == 'text') 
                    {
                        $CustomerPackage['title'] = $packageItem->detail->title;
                    }

                    $CustomerPackage['order_id'] = $id;
                    $CustomerPackage['package_item_id'] = $packageItem->id;
                    $CustomerPackage['item_id'] = $item->id;
                    $CustomerPackage['price'] = $item->price;
                    $CustomerPackage['type'] = $item->type;
                    $CustomerPackage['times'] = $packageItem->times;
                    $CustomerPackage['time_used'] = 0; // not sort out on previous time use
                    $CustomerPackage['time_remain'] = $packageItem->times;
                    $CustomerPackage['expiry_date'] = date("Y-m-d", strtotime("+".$packageItem->duration." Months"));
                    $customerPackageInfo = CustomerPackage::create($CustomerPackage);
                    
                    $customerPackageInfo = CustomerPackage::create($CustomerPackage);

                    $order_customers = OrderCustomer::where('order_id', $id)->get();

                    foreach($order_customers as $order_customer) {
                        $order_customer_info['order_id'] = $order_customer->order_id;
                        $order_customer_info['customer_package_id'] = $customerPackageInfo->id;
                        $order_customer_info['customer_id'] = $order_customer->customer_id;
                        CustomerLinkPackage::create($order_customer_info);
                    }

                }

            }
        }


        if($request->input('payment_option') != 'no_payment') {
            $payment_method = PaymentMethod::with(['detail'])->where('id', $request->input('payment_method_id'))->first();

            $order_transaction['order_id'] = $id;
            $order_transaction['payment_method_id'] = $request->input('payment_method_id');
            $order_transaction['payment_method'] = $payment_method->detail->title;
            $order_transaction['payment_amount'] = $request->input('payment_amount');

            OrderTransaction::create($order_transaction);
        }

        // $this->pdf_save($id);

        return response()->json([
            'message' => 'success',
            'success_message'    => 'update success'
        ]);
    }

    public function destroy(string $id)
    {
        if($this->permission_type['order_delete'] != true) {
            return response()->json(['success' => false]);
        }
        OrderTransaction::where('order_id', $id)->delete();
        OrderCustomer::where('order_id', $id)->delete();
        OrderItem::where('order_id', $id)->delete();
        OrderItemDetail::where('order_id', $id)->delete();
        OrderHistory::where('order_id', $id)->delete();
        
        CustomerPackages::where('order_id', $id)->delete();
        CustomerLinkPackage::where('order_id', $id)->delete();
        
        $response = Order::find($id)->delete();

        if($response) {
            return response()->json([
                'message'  => 'success',
            ]);
        }

        return response()->json([
            'message'  => 'fail',
            'id'    => $id,
            'permission'    => $this->permission_type['order_delete']
        ]);
    }

    public function pdf_view(string $id)
    {
        $pdf = $this->pdf_data($id);

        return $pdf->stream();
        // return view('files.invoice', compact(['data']));

    }

    public function pdf_data(string $id) {
        $order = Order::with(['items', 'order_customers', 'transactions'])->where('id', $id)->first();
        $total_payment = OrderTransaction::where('order_id', $id)->sum('payment_amount');

        $invoice = Setting::where('type', 'invoice')->get();

        $pdf = Pdf::loadView('files.invoice', [
            'order'     => $order,
            'order_id'  => $id,
            'invoice'    => $invoice,
            'total_payment' => $total_payment
        ]);

        return $pdf;
    }

    public function pdf_save(string $id) {
        
        $pdf = $this->pdf_data($id);

        $pdf->save('sales/orders/order_'.$id.'.pdf');
    }

    public function pdf_download(string $id) {
        $pdf = $this->pdf_data($id);

        return $pdf->download('order_'.$id.'.pdf');
    }

}
